<?php
/**
 * BuddyPress XProfile Classes.
 *
 */

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * Member Types xprofile field type.
 *
 * @since 2.0.0
 */
class YOUZIFY_XProfile_Field_Type_MemberType extends BP_XProfile_Field_Type {

    /**
     * Constructor for the selectbox field type.
     *
     * @since 2.0.0
     */
    public function __construct() {

        parent::__construct();

        $this->is_multiple = youzify_member_types_option( 'youzify_member_type_multiple', 'off' );

        if ( $this->is_multiple == 'off' ) {
            $this->category = _x( 'Single Fields', 'xprofile field type category', 'youzify-member-types' );
            $this->supports_options = false;
            $this->supports_multiple_defaults = false;
        } else {
            $this->category = _x( 'Multi Fields', 'xprofile field type category', 'buddypress' );
            $this->supports_multiple_defaults = true;
            $this->supports_options           = false;
        }

        $this->name     = _x( 'Member Types', 'xprofile field type', 'youzify-member-types' );

        $this->accepts_null_value = true;

        $this->set_format( '/^.+$/', 'replace' );

        /**
         * Fires inside __construct() method for BP_XProfile_Field_Type_Selectbox class.
         *
         * @since 2.0.0
         *
         * @param BP_XProfile_Field_Type_Selectbox $this Current instance of
         *                                               the field type select box.
         */
        do_action( 'bp_xprofile_field_type_member_types', $this );
    }

    /**
     * Output the edit field HTML for this field type.
     *
     * Must be used inside the {@link bp_profile_fields()} template loop.
     *
     * @since 2.0.0
     *
     * @param array $raw_properties Optional key/value array of
     *                              {@link http://dev.w3.org/html5/markup/select.html permitted attributes}
     *                              that you want to add.
     */
    public function edit_field_html( array $raw_properties = array() ) {

        // User_id is a special optional parameter that we pass to
        // {@link bp_the_profile_field_options()}.
        if ( isset( $raw_properties['user_id'] ) ) {
            $user_id = (int) $raw_properties['user_id'];
            unset( $raw_properties['user_id'] );
        } else {
            $user_id = bp_displayed_user_id();
        } ?>

        <legend id="<?php bp_the_profile_field_input_name(); ?>-1">
            <?php bp_the_profile_field_name(); ?>
            <?php bp_the_profile_field_required_label(); ?>
        </legend>

        <?php

        /** This action is documented in bp-xprofile/bp-xprofile-classes */
        do_action( bp_get_the_profile_field_errors_action() );?>
        <?php if ( $this->is_multiple == 'off' ) : ?>
        <select <?php echo $this->get_edit_field_html_elements( $raw_properties ); ?> aria-labelledby="<?php bp_the_profile_field_input_name(); ?>-1" aria-describedby="<?php bp_the_profile_field_input_name(); ?>-3">
            <?php bp_the_profile_field_options( array( 'user_id' => $user_id ) ); ?>
        </select>
        <?php endif; ?>
        <?php if ( $this->is_multiple == 'on' ) : ?>
            <?php bp_the_profile_field_options( array(
                'user_id' => $user_id
            ) ); ?>
        <?php endif; ?>
        <?php if ( bp_get_the_profile_field_description() ) : ?>
            <p class="description" id="<?php bp_the_profile_field_input_name(); ?>-3"><?php bp_the_profile_field_description(); ?></p>
        <?php endif; ?>
        <?php
    }

    /**
     * Output the edit field options HTML for this field type.
     *
     * BuddyPress considers a field's "options" to be, for example, the items in a selectbox.
     * These are stored separately in the database, and their templating is handled separately.
     *
     * This templating is separate from {@link BP_XProfile_Field_Type::edit_field_html()} because
     * it's also used in the wp-admin screens when creating new fields, and for backwards compatibility.
     *
     * Must be used inside the {@link bp_profile_fields()} template loop.
     *
     * @since 2.0.0
     *
     * @param array $args Optional. The arguments passed to {@link bp_the_profile_field_options()}.
     */
    public function edit_field_options_html( array $args = array() ) {

        if  ( $this->is_multiple == 'off' ) {
            $this->get_single_field_options( $args );
        } else {
            $this->get_multiple_field_options( $args );
        }

    }

    /**
     * Get Single Field.
     */
    function get_single_field_options( $args = array() ) {

        $original_option_values = maybe_unserialize( BP_XProfile_ProfileData::get_value_byid( $this->field_obj->id, $args['user_id'] ) );

        $html = '';

        if ( 'on' == youzify_member_types_option( 'youzify_allow_no_member_type', 'off' ) ) {
            $html    = '<option value="">' . /* translators: no option picked in select box */ esc_html__( '----', 'youzify-member-types' ) . '</option>';
        }

        if ( empty( $original_option_values ) && !empty( $_POST['field_' . $this->field_obj->id] ) ) {
            $original_option_values = sanitize_text_field(  $_POST['field_' . $this->field_obj->id] );
        }

        $option_values = ( $original_option_values ) ? (array) $original_option_values : array();

        $member_types = youzify_get_registration_member_types();

        foreach ( $member_types as $type_id => $member_type ) {

            // print_r( $member_type );

            // Init Selected
            $selected = '';

            // Run the allowed option name through the before_save filter, so we'll be sure to get a match
            $allowed_options = xprofile_sanitize_data_value_before_save( $type_id , false, false );

            // First, check to see whether the user-entered value matches
            if ( in_array( $allowed_options, (array) $option_values ) ) {
                $selected = ' selected="selected"';
            }

            // Get Member Type ID.
            // $type_id = isset( $member_type['id'] ) ? $member_type['id'] : youzify_get_member_type_id( $member_type['singular'] );

            $html .= apply_filters( 'bp_get_the_profile_field_options_member_type', '<option' . $selected . ' value="' . esc_attr( $type_id ) . '">' . $member_type->labels['singular_name'] . '</option>', $type_id, $member_type->labels['singular_name'] , $this->field_obj->id, $selected );

        }

        echo $html;
    }

    /**
     * Get Multiple Field.
     */
    function get_multiple_field_options( $args = array() ) {

        $original_option_values = maybe_unserialize( BP_XProfile_ProfileData::get_value_byid( $this->field_obj->id, $args['user_id'] ) );

        $html = '';

        if ( empty( $original_option_values ) && !empty( $_POST['field_' . $this->field_obj->id] ) ) {
            $original_option_values = sanitize_text_field(  $_POST['field_' . $this->field_obj->id] );
        }

        $option_values = ( $original_option_values ) ? (array) $original_option_values : array();

        $member_types = youzify_get_registration_member_types();

        foreach ( $member_types as $type_id => $member_type ) {

            // Init Selected
            $selected = '';

            // Run the allowed option name through the before_save filter, so we'll be sure to get a match
            $allowed_options = xprofile_sanitize_data_value_before_save( $type_id, false, false );

            // First, check to see whether the user-entered value matches
            if ( in_array( $allowed_options, (array) $option_values ) ) {
                $selected = ' checked="checked"';
            }

            $html .= sprintf( '<label for="%3$s" class="option-label"><input %1$s type="checkbox" name="%2$s" id="%3$s" value="%4$s">%5$s</label>',
                $selected,
                esc_attr( bp_get_the_profile_field_input_name() . '[]' ),
                esc_attr( "field_$type_id" ),
                esc_attr( stripslashes( $type_id ) ),
                esc_html( stripslashes( $member_type->labels['singular_name'] ) )
            );

        }

        printf( '<div id="%1$s" class="input-options checkbox-options">%2$s</div>',
            esc_attr( 'field_' . $this->field_obj->id ),
            $html
        );
    }

    /**
     * Output HTML for this field type on the wp-admin Profile Fields screen.
     *
     * Must be used inside the {@link bp_profile_fields()} template loop.
     *
     * @since 2.0.0
     *
     * @param array $raw_properties Optional key/value array of permitted attributes that you want to add.
     */
    public function admin_field_html( array $raw_properties = array() ) {

        if ( $this->is_multiple == 'on' ) {
            echo '<style type="text/css"> .member_types label { margin-right: 15px; }</style>';
            bp_the_profile_field_options();
            return;
        }

        ?>

        <label for="<?php bp_the_profile_field_input_name(); ?>" class="screen-reader-text"><?php
            /* translators: accessibility text */
            esc_html_e( 'Select', 'youzify-member-types' );
        ?></label>
        <select <?php echo $this->get_edit_field_html_elements( $raw_properties ); ?>>
            <?php bp_the_profile_field_options(); ?>
        </select>

        <?php
    }

    /**
     * Output HTML for this field type's children options on the wp-admin Profile Fields "Add Field" and "Edit Field" screens.
     *
     * Must be used inside the {@link bp_profile_fields()} template loop.
     *
     * @since 2.0.0
     *
     * @param BP_XProfile_Field $current_field The current profile field on the add/edit screen.
     * @param string            $control_type  Optional. HTML input type used to render the current
     *                                         field's child options.
     */
    public function admin_new_field_html( BP_XProfile_Field $current_field, $control_type = '' ) {
        $field_type = $this->is_multiple == 'on' ? 'checkbox' : 'radio';
        parent::admin_new_field_html( $current_field, $field_type );
    }

}
