<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

//================================================================================
// QUERYING
//================================================================================

/**
 * Add new query variables in the list of public query variables, 
 * so that we can use it in our custom URLs.
 * https://codex.wordpress.org/Function_Reference/get_query_var
 */
function add_public_query_vars_for_aeiforia_projects( $qvars ) {
    // Back-end AEIFORIA Project listings
    $qvars[] = "aeiforia_project_meta_query";
    // Front-end AEIFORIA Project listings
    $myvars = [
        "viewing",
        "keywords",
        "apc", # Aeiforia Project Category
        "asp", # Aeiforia Sustainability Pillar
        "project_status",
        "project_universities",
        "project_regions",
        "project_stakeholders",
        "project_fs",
        "project_sdgs"
    ];
    foreach ( $myvars as $key ) {
        if ( ! array_key_exists( $key, $qvars ) ) {
            $qvars[] = $key;
        }
    }

    return $qvars;
}
add_filter( 'query_vars', 'add_public_query_vars_for_aeiforia_projects' );

/**
 * Manipulate query before quering 
 * and map 'aeiforia_project_meta_query' custom query argument 
 * to the 'meta_query' WordPress's query argument.
 */
function aeiforia_project_pre_get_posts( $query ) {

    // if( function_exists('get_current_screen') && get_current_screen()->parent_base == 'edit' && is_admin() && $query->is_main_query() ) {
    //      //$query->query_vars['suppress_filters'] = true;
    //  }

    /**
     * If we're viewing the administration panel, 
     * and we're quering for "aeiforia-project" custom post type, 
     * and there's a "aeiforia_project_meta_query" custom query argument present, 
     * and its value is not empty.
     */
    if( is_admin() && get_query_var( 'post_type' ) == 'aeiforia-project' && ! empty( get_query_var( 'aeiforia_project_meta_query' ) ) && ! empty( $query->query_vars['aeiforia_project_meta_query'][0]['value'] ) && $query->is_main_query() ) {

        // Map 'aeiforia_project_meta_query' to 'meta_query'
        $query->set( 'meta_query', get_query_var( 'aeiforia_project_meta_query' ) );

        // echo '<pre>';
        // var_dump($query);
        // echo '</pre>';
        // die();

        // echo '<pre>';
        // print_r(get_query_var( 'aeiforia_project_meta_query' ));
        // echo '</pre>';
        // $temp = get_query_var( 'aeiforia_project_meta_query' );
        // echo $temp[0]["key"]; // assigned_to
        // die();

        // echo $query->query_vars['post_type'];
        // echo $query->query_vars['aeiforia_project_meta_query'][0]['key'];
        // echo '<pre>';
        // var_dump(get_query_var( 'aeiforia_project_meta_query[0][key]' ));
        // echo '</pre>';
        // die();

        //get_query_var( 'aeiforia_project_meta_query' )
        // array(1) {
        //     [0]=>
        //     array(1) {
        //       ["key"]=>
        //       string(11) "assigned_to"
        //     }
        //   }

        // if( !get_query_var( 'aeiforia_project_meta_query[0][value]' ) ) {
        //     echo "Yeah";
        //     // urldecode
        //     unset( $query->query_vars['aeiforia_project_meta_query[0][key]'] );
        //     remove_query_arg( 'aeiforia_project_meta_query', $query );
        //     remove_query_arg( urlencode('aeiforia_project_meta_query[0][key]'), $query );
        // }
    }

}
add_action( 'pre_get_posts', 'aeiforia_project_pre_get_posts' );

/**
 * Filter by custom taxonomy term and/or keyword, if present.
 * Choose from different query arguments depending on
 * the passed terms and keyword.
 * 
 * @return array $query_args An array of arguments to pass to WP_Query()
 */
function getAeiforiaQueryArguments($project_category_term = null, $sustainability_pillar_terms = null, $project_status_terms = null, $project_university_terms = null, $project_region_terms = null, $project_stakeholder_terms = null, $project_funding_sources_terms = null, $project_sdg_terms = null, $keywords = null ) {

    // Get all projects by default
    $query_args = [
        'post_type' => ['aeiforia-project'],
        //'post_status' => ['publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit'],
        'post_status' => 'publish',
        'orderby'     => 'date',
        'numberposts' => -1,
        //'numberposts' => '5',
        'posts_per_page' => get_option('posts_per_page'),
        'paged' => ( get_query_var( 'paged' ) ) ? absint( get_query_var( 'paged' ) ) : 1,
    ];

    $tax_query = [];

    if ( !empty($project_category_term) ) {
        $tax_query[] = [
            'taxonomy' => 'aeiforia-project-category',
            'field' => 'term_id',
            'terms' => [$project_category_term],
            'operator' => 'IN',
        ];
    }

    if ( !empty($sustainability_pillar_terms) ) {
        $tax_query[] = [
            'taxonomy' => 'aeiforia-sustainability-pillar',
            'field' => 'term_id',
            'terms' => $sustainability_pillar_terms,
            'operator' => 'IN',
        ];
    }

    if ( !empty($project_status_terms) ) {
        $tax_query[] = [
            'taxonomy' => 'aeiforia-project-status',
            'field' => 'term_id',
            'terms' => $project_status_terms,
            'operator' => 'IN',
        ];
    }

    if ( !empty($project_university_terms) ) {
        $tax_query[] = [
            'taxonomy' => 'aeiforia-erua-university',
            'field' => 'term_id',
            'terms' => $project_university_terms,
            'operator' => 'IN',
        ];
    }

    if ( !empty($project_region_terms) ) {
        $tax_query[] = [
            'taxonomy' => 'aeiforia-erua-region',
            'field' => 'term_id',
            'terms' => $project_region_terms,
            'operator' => 'IN',
        ];
    }

    if ( !empty($project_stakeholder_terms) ) {
        $tax_query[] = [
            'taxonomy' => 'aeiforia-erua-stakeholder',
            'field' => 'term_id',
            'terms' => $project_stakeholder_terms,
            'operator' => 'IN',
        ];
    }

    if ( !empty($project_funding_sources_terms) ) {
        $tax_query[] = [
            'taxonomy' => 'aeiforia-funding-source',
            'field' => 'term_id',
            'terms' => $project_funding_sources_terms,
            'operator' => 'IN',
        ];
    }

    if ( !empty($project_sdg_terms) ) {
        $tax_query[] = [
            'taxonomy' => 'aeiforia-sdg',
            'field' => 'term_id',
            'terms' => $project_sdg_terms,
            'operator' => 'IN',
        ];
    }

    /**
     * Important/Beware: ACF serialized meta data, meta querying with LIKE comparison = slow performance
     * Not all ACF meta is serialized in the database, so manually check the wp_postmeta table for the meta_value.
     * 
     * https://wordpress.stackexchange.com/questions/298156/meta-query-array-in-values-returns-an-error
     * https://wordpress.stackexchange.com/questions/16709/meta-query-with-meta-values-as-serialize-arrays
     * https://stackoverflow.com/questions/26364378/wp-query-when-meta-value-saved-as-serialized-array
     * https://support.advancedcustomfields.com/forums/topic/using-compare-in-operator-for-meta_query/
     */
    //$meta_query = [];

    // if ( !empty($project_funding_sources_terms) ) {
    //     $meta_query[] = [
    //         'key' => 'aeiforia_project_funding_source',
    //         'value' => $project_funding_sources_terms,
    //         'compare' => 'IN', # 'IN' for arrays or 'LIKE' for strings
    //     ];
    // }

    if ( !empty($keywords) ) {
        /**
         * Note: Prepending a keyword with a hyphen will exclude posts matching that keyword.
         * E.g., 'pillow -sofa' will return posts containing ‘pillow’ but not ‘sofa’.
         * 
         * https://developer.wordpress.org/reference/classes/wp_query/#search-parameters
         */
        $query_args['s'] = sanitize_text_field($keywords);
    }

    if ( !empty($tax_query) ) {
        if ( count($tax_query) > 1 ) $tax_query = ['relation' => 'AND'] + $tax_query;
        $query_args['tax_query'] = $tax_query;
    }

    // if ( !empty($meta_query) ) {
    //     if ( count($meta_query) > 1 ) $meta_query = ['relation' => 'OR'] + $meta_query;
    //     $query_args['meta_query'] = $meta_query;
    // }

    return $query_args;

}

//================================================================================
// LISTINGS SCREEN (edit.php) CUSTOMIZATION - COLUMNS, FILTERS etc.
//================================================================================

/**
 * Customize views @edit.php?post_type=aeiforia-project
 * More @ https://codex.wordpress.org/Plugin_API/Filter_Reference/views_edit-post
 */
function remove_views_from_aeiforia_projects($views) {  
    $user = wp_get_current_user();
    if ( in_array( 'administrator', (array) $user->roles ) ) {
        unset($views['mine']);
    }
    return $views;
}
add_filter('views_edit-aeiforia-project', 'remove_views_from_aeiforia_projects');

/**
 * Register columns for the back-end listings of the "aeiforia-project" custom post type.
 * https://codex.wordpress.org/Plugin_API/Filter_Reference/manage_$post_type_posts_columns
 */
function register_backend_aeiforia_project_columns($columns)
{
    $columns['status'] = __('Status', 'erua');
    // return [
    //     'cb' => '<input type="checkbox" />',
    //     'title' => __('Title', 'erua'),
    //     //'author' => __('Author', 'erua'),
    //     //'categories' => __('Categories', 'erua'),
    //     'applicant' => __('Applicant', 'erua'),
    //     'email' => __('Email', 'erua'),
    //     'topic' => __('Topic', 'erua'),
    //     'type' => __('Type', 'erua'),
    //     'stage' => __('Stage', 'erua'),
    //     'countries' => __('Countries', 'erua'),
    //     'languages' => __('Languages', 'erua'),
    //     'status' => __('Status', 'erua'),
    //     'assignee' => __('Assignee', 'erua'),
    //     'comments' => '<span class="vers comment-grey-bubble" title="' . esc_attr__('Comments', 'erua') . '"><span class="screen-reader-text">' . __('Comments', 'erua') . '</span></span>',
    //     'date' => __('Date', 'erua'),
    // ];
    return $columns;
}
add_filter('manage_aeiforia-project_posts_columns', 'register_backend_aeiforia_project_columns');

/**
 * Create the content of the custom columns
 * that were configured with register_backend_aeiforia_project_columns()
 * http://justintadlock.com/archives/2011/06/27/custom-columns-for-custom-post-types
 * https://wordpress.stackexchange.com/questions/253640/adding-custom-columns-to-custom-post-types
 * https://wpsmackdown.com/easy-filter-posts-wordpress-admin/
 * https://codex.wordpress.org/WordPress_Query_Vars
 * https://wordpress.stackexchange.com/questions/212519/filter-by-custom-field-in-custom-post-type-on-admin-page
 */
function manage_aeiforia_project_backend_custom_columns($column, $post_id)
{
    global $post;

    switch ($column) {

        /* If displaying the "status" column. */
        case 'status':

            /* Get the stages for the post. */
            $terms = get_the_terms($post_id, 'aeiforia-project-status');

            /* If no terms were found, output a default message. */
            if ( empty($terms) ) {
                echo __('Unknown', 'erua');
            }
            /* If there is a status */
            else {
                $out = array();
                foreach ($terms as $term) {
                    // "Active"
                    if ($term->term_id == 44) {
                        $class_name = 'project-status active-status';
                    }
                    // "Closed"
                    else if ($term->term_id == 45) {
                        $class_name = 'project-status closed-status';
                    }
                    else {
                        $class_name = 'project-status generic-status';
                    }

                    $out[] = sprintf('<a href="%s" class="%s">%s</a>',
                        esc_url(add_query_arg(['post_type' => $post->post_type, 'aeiforia-project-status' => $term->slug], 'edit.php')),
                        $class_name,
                        esc_html(sanitize_term_field('name', $term->name, $term->term_id, 'aeiforia-project-status', 'display'))
                    );
                }
                echo join(', ', $out);
            }

            break;

        /* Just break out of the switch statement for everything else. */
        default:
            break;
    }
}
add_action('manage_aeiforia-project_posts_custom_column', 'manage_aeiforia_project_backend_custom_columns', 10, 2);

/**
 * Add filters based on custom taxonomies,
 * for the project listings (@ edit.php).
 * https://generatewp.com/filtering-posts-by-taxonomies-in-the-dashboard/
 */
function create_backend_filters_for_aeiforia_projects($post_type, $which)
{

    // Apply this only on a specific post type
    if ('aeiforia-project' === $post_type) {

        /**
         * A list of taxonomy slugs to filter by
         * Note: Private and draft posts will be displayed 
         * but won't be counted in the taxonomy's terms.
         */
        $taxonomies = [
            'aeiforia-project-category',
            'aeiforia-sustainability-pillar',
            'aeiforia-project-status',
            'aeiforia-erua-university',
            'aeiforia-erua-region',
            'aeiforia-erua-stakeholder',
            'aeiforia-funding-source',
            'aeiforia-sdg'
        ];

        foreach ($taxonomies as $taxonomy_slug) {

            // Retrieve taxonomy data
            $taxonomy_obj = get_taxonomy($taxonomy_slug);
            $taxonomy_name = $taxonomy_obj->labels->name;
            $taxonomy_is_hierarchical = $taxonomy_obj->hierarchical;

            if ( $taxonomy_is_hierarchical ) {

                /**
                 * Retrieve parent terms
                 */
                $top_level_terms = get_terms( [
                    'taxonomy'      => $taxonomy_slug,
                    'parent'        => '0',
                    'hide_empty'    => false,
                    'suppress_filters' => false
                ] );

                // Display filter HTML
                echo "<select name='{$taxonomy_slug}' id='{$taxonomy_slug}' class='postform'>";
                echo '<option value="">' . sprintf( esc_html__('All %s', 'erua'), $taxonomy_name ) . '</option>';
                foreach ( $top_level_terms as $top_level_term ) {

                    // View all projects with an awaiting nature (pending, waiting for x's reply etc.) of status
                    $query = get_posts(
                        [
                            'post_type' => $post_type,
                            'fields' => 'ids',
                            'post_status' => ['publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit'],
                            'numberposts' => -1,
                            'tax_query' => [
                                [
                                    'taxonomy' => $taxonomy_slug,
                                    'field' => 'slug',
                                    'terms' => $top_level_term->slug,
                                    'operator' => 'IN'
                                ]
                            ]
                        ]
                    );
                    wp_reset_postdata();
                    $count = count($query);

                    printf(
                        '<option value="%1$s" %2$s>%3$s (%4$s)</option>',
                        $top_level_term->slug,
                        ((isset($_GET[$taxonomy_slug]) && ($_GET[$taxonomy_slug] == $top_level_term->slug)) ? ' selected="selected"' : ''),
                        $top_level_term->name,
                        $count //$top_level_term->count
                    );

                    $top_term_id = $top_level_term->term_id;

                    $second_level_terms = get_terms( array(
                        'taxonomy' => $taxonomy_slug, # you could also use $taxonomy as defined in the first lines
                        'child_of' => $top_term_id,
                        'parent' => $top_term_id, # disable this line to see more child elements (child-child-child-terms)
                        'hide_empty' => false,
                    ) );
                    foreach ($second_level_terms as $second_level_term) {

                        $query = get_posts(
                            [
                                'post_type' => $post_type,
                                'fields' => 'ids',
                                'post_status' => ['publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit'],
                                'numberposts' => -1,
                                'tax_query' => [
                                    [
                                        'taxonomy' => $taxonomy_slug,
                                        'field' => 'slug',
                                        'terms' => $second_level_term->slug,
                                        'operator' => 'IN'
                                    ]
                                ]
                            ]
                        );
                        wp_reset_postdata();
                        $count = count($query);

                        printf(
                            '<option value="%1$s" %2$s>%3$s (%4$s)</option>',
                            $second_level_term->slug,
                            ((isset($_GET[$taxonomy_slug]) && ($_GET[$taxonomy_slug] == $second_level_term->slug)) ? ' selected="selected"' : ''),
                            '&nbsp;&nbsp;&nbsp;&nbsp;' . $second_level_term->name,
                            $count//$second_level_term->count
                        );
                    }
                }
                echo '</select>';
            }
            else {

                /**
                 * Retrieve parent terms
                 */
                $top_level_terms = get_terms( [
                    'taxonomy'      => $taxonomy_slug,
                    'parent'        => '0',
                    'hide_empty'    => false,
                    'suppress_filters' => false
                ] );

                // Display filter HTML
                echo "<select name='{$taxonomy_slug}' id='{$taxonomy_slug}' class='postform'>";
                echo '<option value="">' . sprintf( esc_html__('Show All %s', 'erua'), $taxonomy_name ) . '</option>';
                foreach ( $top_level_terms as $top_level_term ) {

                    // View all projects with an awaiting nature (pending, waiting for x's reply etc.) of status
                    $query = get_posts(
                        [
                            'post_type' => $post_type,
                            'fields' => 'ids',
                            'post_status' => ['publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit'],
                            'numberposts' => -1,
                            'tax_query' => [
                                [
                                    'taxonomy' => $taxonomy_slug,
                                    'field' => 'slug',
                                    'terms' => $top_level_term->slug,
                                    'operator' => 'IN'
                                ]
                            ]
                        ]
                    );
                    wp_reset_postdata();
                    $count = count($query);

                    printf(
                        '<option value="%1$s" %2$s>%3$s (%4$s)</option>',
                        $top_level_term->slug,
                        ((isset($_GET[$taxonomy_slug]) && ($_GET[$taxonomy_slug] == $top_level_term->slug)) ? ' selected="selected"' : ''),
                        $top_level_term->name,
                        $count //$top_level_term->count
                    );
                }
                echo '</select>';
            }
        }

    }

}
add_action('restrict_manage_posts', 'create_backend_filters_for_aeiforia_projects', 10, 2);

//================================================================================
// BACK-END FIELD HANDLING
//================================================================================

/**
 * Hooking into 'acf/save_post' with a high priority such as 20,
 * ensures that the $_POST data have already been saved,
 * and therefore we can get/access all custom fields of the post.
 * Alternatively we may use a priority lower than 10,
 * to manipulate the data before it gets saved.
 * 
 * https://www.advancedcustomfields.com/resources/acf-save_post/
 */
function aeiforia_project_after_save_post( $post_id ) {

    if ( is_admin() && get_post_type($post_id) !== 'aeiforia-project' ) {
        return;
    }

    // Update aeiforia-sdg taxonomy terms based on the SDG field data
    if ( isset($_POST['acf']['field_65197ab6a632d']) ) {
        $sdg_terms = $_POST['acf']['field_65197ab6a632d'];
        $sdg_terms_ids = array_map('intval', $sdg_terms);
        wp_set_object_terms( (int) $post_id, $sdg_terms_ids, 'aeiforia-sdg', false );
    }
}
add_action('acf/save_post', 'aeiforia_project_after_save_post', 20);

/**
 * Load field choices for SDGs and other fields.
 *
 * https://www.advancedcustomfields.com/resources/acf-load_value/
 */
function aeiforia_project_load_field_choices( $value, $post_id, $field ) {

    $sdg_terms = get_the_terms($post_id, 'aeiforia-sdg');

    if ( $sdg_terms ) {
        // Reset values
        $value = [];

        foreach ( $sdg_terms as $key => $sdg ) {
            $value[] = strval($sdg->term_id);
        }

        // echo '<pre>';
        // var_dump($value);
        // echo '</pre>';

    }

    return $value;
}
add_filter('acf/load_value/key=field_65197ab6a632d', 'aeiforia_project_load_field_choices', 10, 3); # SDGs

//================================================================================
// BUBBLE NOTIFICATIONS
//================================================================================

/**
 * Notification "bubble" with pending/open Projects count
 */
add_action('admin_menu', function () {
    global $menu;
    //$count_posts = wp_count_posts('aeiforia-project');
    //$count = $count_posts->pending;

    // View all projects with an awaiting nature (pending, waiting for x's reply etc.) of status
    $projects_query = get_posts(
        [
            'post_type' => 'aeiforia-project',
            'fields' => 'ids',
            //'post_status' => 'any', // Published, Pending etc.
            //'post_status' => ['publish', 'pending', 'draft', 'auto-draft', 'future', 'private', 'inherit', 'trash'],
            //'post_status' => ['publish', 'pending', 'draft', 'future', 'private', 'inherit'],
            'post_status' => ['pending', 'draft'],
            'numberposts' => -1,
            // All projects that are not solved or closed i.e. pending or open
            // 'tax_query' => [
            //     [
            //         'taxonomy' => 'aeiforia-project-status',
            //         'field' => 'slug',
            //         'terms' => ['closed', 'solved'],
            //         'operator' => 'NOT IN',
            //     ],
            // ],
        ]
    );
    if (!empty($projects_query)) wp_reset_postdata();
    $count = count($projects_query);
    /* echo '<pre>';
    var_dump($projects_query);
    echo '</pre>';
    die(); */

    $menu_item = wp_list_filter(
        $menu,
        [2 => 'edit.php?post_type=aeiforia-project'] // 2 is the position of an array item which contains URL, it will always be 2!
    );
    if ( !empty($menu_item) && $count > 0 ) {
        $menu_item_position = key($menu_item); // get the array key (position) of the element
        $menu[$menu_item_position][0] .= ' <span class="awaiting-mod" title="' . $count . ' Projects require actions">' . $count . '</span>';
    }
});
